#include <hidef.h> 
#include "derivative.h" 
#include "EEEfunctions.h"

#define FACCERR	0x10	           // define access error bit
#define FPVIOL  0x20	           // define protocol violation bit
#define FCCF	  0x40	           // define flash command complete flag
#define FCBEF   0x80	           // define flash command buffer empty flag


/**************************************************************
*    Function:     CopyFLASHfuncsTORAM(*src,*dest,size)
*    Parameters: *src  - pointer to the ROM memory location of the funciton
*                *dest - pointer to the RAM memory location of the function
*                size  - the size of the function
*    Return:     none
*
*    The application can not run from Flash and write to Flash at the
*      same time.  This routine copies the function that writes FLASH 
*      into RAM so that it can execute from there.
***************************************************************/

void CopyFLASHfuncsToRAM(unsigned char *src, unsigned char *dest, unsigned char size)
{
	while(size != 0)
	{
		*dest++ = *src++;		
		size--;
	} 
}


/**************************************************************
*    Function:     ProgramEEEBytes(*EEEaddress,*rambuffer,numbytes)
*    Parameters: *EEEaddress - pointer to the Flash location where the variable is stored
*                *rambuffer  - pointer to the RAM buffer that has the data
*                numbytes    - number of bytes to be stored
*    Return:     status      - Indication if the Programming was successful or not
*
*    The function sets up and makes a call to the Program/Erase function.
*      The P/E function is set in Burst mode which is a quicker way to  
*      program more than one byte at a time.  See Section 4.5 of the
*      9S08QE8 data sheet for details.
***************************************************************/

uchar ProgramEEEBytes(uchar *EEEaddress, uchar *rambuffer, uchar numbytes)
{
	uchar status;
	
	status = ProgEraseFlash(BURST_PROGRAM, EEEaddress, rambuffer, numbytes);
		
	return(status);
}


/**************************************************************
*    Function:     EraseEEEPage(*EEEaddress)
*    Parameters: *EEEaddress - pointer to the Flash location to be erased
*    Return:     status      - Indication if the Erasing was successful or not
*
*    The function sets up and makes a call to the Program/Erase function.
*      The P/E function is set in Erase Page mode which erased a single  
*      512Byte page of memory.  See Section 4.5 of the 9S08QE8 data sheet
*      for details.
***************************************************************/

uchar EraseEEEPage(uchar *EEEaddress)
{
	uchar status;
	
	status = ProgEraseFlash((uchar)PAGE_ERASE, EEEaddress, EEEaddress, 1);
			
	return(status);	
}  


/**************************************************************
*    Function:     ProgEraseFlash(cmd, *progaddress, *bufferaddress, buffersize)
*    Parameters: cmd - Command that the Flash controller should execute
*                *progaddress - pointer to the Flash location to be erased
*                *bufferaddress - pointer to the RAM buffer that has the data
*                buffersize  - size of data to process
*    Return:     status - Indication if the Flash routine was successful or not
*
*    The function controls the Flash programming and erasing.
*      There is a strict process flow that the routine must follow.  
*      See Section 4.5 of the 9S08QE8 reference manual for details.
*      This function must operate out of RAM and is copied into
*      RAM at start-up.  The routine uses 82bytes of code space
*      and 2 bytes of data.
***************************************************************/

#pragma CODE_SEG FLASH_ROUTINES    //Places the routine in a specific locatio of ROM so it can be copied to RAM

unsigned char ProgEraseFlash(unsigned char cmd, unsigned char *progaddress, unsigned char *bufferaddress, signed char buffersize)
{
	unsigned char i,ccr;
	
    /* !!! make sure interrupts are disabled !!! */
	asm TPA;				                        // get current state of condition code register
	asm STA ccr;		                 	        // ... and save it
	asm SEI;				                        // make sure interrupts are disabled				  	
	
	FSTAT = (FACCERR | FPVIOL);		   	            // clear the FACCERR and FPVIOL flags					
	buffersize--;
	while ( buffersize >= 0 )
	{
		while (!(FSTAT & FCBEF))
		    ;
		*progaddress++ = *bufferaddress++;
		FCMD = cmd;				                    // send prog / erase command
		FSTAT = FCBEF;			                    // start command buffer.
		buffersize--;
	}

	while ( !(FSTAT & FCCF) )
	    ;                                           // wait until the write/erase has completed

	i = PASS;		
    if ( (FSTAT & (FACCERR|FPVIOL))!= 0 )
    	i = FAIL;
    
    
	asm LDA ccr;	                     	        // restore the interrupt state
	asm TAP;			                            // restore state of condition code register
		
	return(i);    	
}

#pragma CODE_SEG DEFAULT
